﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Linq;
    using System.Threading.Tasks;
    using System.Collections.Generic;

    using Domain.Services;

    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;

    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels;
    using Shared.UserModels.PatientExcel;
    using Shared.UserModels.ExcelUploads;
    using Shared.Library.Enums;

    using Utilities;
    using Hims.Api.Models;
    using Hims.Shared.UserModels.UserExcel;
    using Hims.Infrastructure.Services;
    using Hims.Shared.UserModels.Filters;
    using AutoMapper;

    /// <summary>
    /// The patient excel history controller.
    /// </summary>
    [Authorize]
    [Route("api/patient-import")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PatientExcelHistoryController : BaseController
    {
        /// <summary>
        /// The excel upload history service.
        /// </summary>
        private readonly IPatientExcelHistoryService patientExcelHistoryService;

        /// <summary>
        /// The user service.
        /// </summary>
        private readonly IPatientService patientsServices;
        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;
        /// <summary>
        /// The mapper.
        /// </summary>
        private readonly IMapper mapper;
        /// <inheritdoc />
        public PatientExcelHistoryController(

            IPatientExcelHistoryService patientExcelHistoryService,
            IPatientService patientsServices,
            IAuditLogService auditLogServices,
              IMapper mapper)
        {
            this.patientExcelHistoryService = patientExcelHistoryService;
            this.patientsServices = patientsServices;
            this.auditLogServices = auditLogServices;
            this.mapper = mapper;
        }

        /// <summary>
        /// The add products from excel async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header"></param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("add-patient-sheet")]
        public async Task<ActionResult> AddpatientFromExcelAsync([FromBody] PatientExcelHistoryModel model, [FromHeader] LocationHeader header)
        {
            model = (PatientExcelHistoryModel)EmptyFilter.Handler(model);
            int locationId = Convert.ToInt32(header.LocationId);
            model.LocationId = locationId;
            var response = 0;
            if (model.Patients.Count == 0)
            {
                return this.ServerError("Invalid sheet");
            }
            var (patientLst, unMatchedList) = await this.AddPatientsAsync(model);
            if (patientLst.Count > 0)
            {
                var excelHistory = new PatientExcelHistoryModel
                {
                    PatientExcelHistoryId = model.PatientExcelHistoryId,
                    SheetName = model.SheetName,  
                    UploadedBy = model.CreatedBy,
                    AddedPatients = string.Join(",", patientLst.Select(m => m.PatientId)),
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                response = await this.patientExcelHistoryService.ModifyExcelUploadAsync(excelHistory);
            }
            if(response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.ExcelSheet,
                    LogFrom = (short)model.RoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"{model.UploadedByName} has uploaded {model.SheetName} on {DateTime.UtcNow.AddMinutes(330).ToString("MM/dd/yyyy hh:mm tt")}",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
            return this.Success(new Tuple<int, List<RawPatientModel>>(response, unMatchedList ?? new List<RawPatientModel>()));
        }


        /// <summary>
        /// The fetch all excel upload history.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch-upload-history")]
        public async Task<ActionResult> FetchAllExcelUploadHistory([FromBody] PatientExcelHistoryModel model)
        {
            model = (PatientExcelHistoryModel)EmptyFilter.Handler(model);
            var response = await this.patientExcelHistoryService.FetchAllAsync(model);
            return this.Success(response);
        }

        /// <summary>
        /// The fetch all excel users async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch-uploaded-patients")]
        public async Task<ActionResult> FetchAllExcelPatientsAsync([FromBody] PatientExcelHistoryModel model)
        {
            model = (PatientExcelHistoryModel)EmptyFilter.Handler(model);
            
                var response = (await this.patientsServices.FetchAsync(new PatientFilterModel())).ToList();
            
            
            var patientArray = model.AddedPatients.Split(",");
            var patientList = new List<PatientModel>();
            for (int i = 0; i < patientArray.Length; i++)
            {
                var patientId = int.Parse(patientArray[i]);
                var patient = response.Find(m => m.PatientId == patientId);
                patientList.Add(patient);
            }
                return this.Success(patientList);
            
            
        }

        /// <summary>
        /// The add pharmacy product async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        private async Task<Tuple<List<PatientModel>, List<RawPatientModel>>> AddPatientsAsync(PatientExcelHistoryModel model)
        {
            var totalPatientList = new List<PatientModel>();
            var returnModel = new List<RawPatientModel>();
            foreach (var patient in model.Patients)
            {               
                var singlePatient = new PatientModel();
                singlePatient.REGNO =patient.UmrNo;
                singlePatient.Salutation = patient.Salutation;
                singlePatient.FullName = CoreFilter.FullName(patient.FirstName, " ", patient.LastName);
                singlePatient.FirstName = patient.FirstName;
                singlePatient.LastName = patient.LastName;
                singlePatient.Email = patient.Email;
                singlePatient.AddressLine2 = patient.Address;
                singlePatient.Gender = char.Parse(patient.Gender);
                singlePatient.DateOfBirth = Convert.ToDateTime(patient.DateOfBirth);
                singlePatient.Age =Convert.ToInt16(patient.Age);
                singlePatient.Mobile = patient.Mobile;
                singlePatient.Zipcode = patient.PinCode;
                singlePatient.CreatedBy = model.CreatedBy;
                singlePatient.RelationType = patient.RelationType;
                singlePatient.FatherOrHusband = patient.RelativeName;
                singlePatient.LocationId = model.LocationId;
                singlePatient.CountryId = 1;
                singlePatient.PaymentStatus = true;
                List<PatientFamilyModel> singlePatientFamily = new List<PatientFamilyModel>();
                if ((patient.RelationType!="")||(patient.RelativeName!="")||(patient.RelativeMobile!=""))
                {
                     singlePatientFamily = new List<PatientFamilyModel> { new PatientFamilyModel { Relation = patient.RelationType, FullName = patient.RelativeName, ContactNo = patient.RelativeMobile } };
                }
                List<PatientEmergencyModel> singleEmergencyPatientInfo = new List<PatientEmergencyModel> { new PatientEmergencyModel { FullName = patient.RelativeName, Relation = patient.RelationType, Mobile = patient.RelativeMobile } };
                                                                                 
                var (accountId, patientId, guid) = await this.patientsServices.AddAsync(singlePatient, singleEmergencyPatientInfo, new List<PatientInsuranceModel>(),  new PatientFamilyModel(), singlePatientFamily ?? new List<PatientFamilyModel>());
                    if (patientId < 0)
                    {
                        returnModel.Add(patient);
                    }
                    else
                    {
                        singlePatient.PatientId = patientId;
                        totalPatientList.Add(singlePatient);
                    }
            }
            
            return new Tuple<List<PatientModel>, List<RawPatientModel>>(totalPatientList, returnModel);
        }

    }
}
